#include "Data_Reader.h"

#include <assert.h>
#include <string.h>
#include <stdio.h>

/* Copyright (C) 2005-2006 Shay Green. Permission is hereby granted, free of
charge, to any person obtaining a copy of this software module and associated
documentation files (the "Software"), to deal in the Software without
restriction, including without limitation the rights to use, copy, modify,
merge, publish, distribute, sublicense, and/or sell copies of the Software, and
to permit persons to whom the Software is furnished to do so, subject to the
following conditions: The above copyright notice and this permission notice
shall be included in all copies or substantial portions of the Software. THE
SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED,
INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A
PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR
COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER
IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE. */

typedef Data_Reader::error_t error_t;

error_t Data_Reader::read( void* p, long s )
{
	long result = read_avail( p, s );
	if ( result != s )
	{
		if ( result >= 0 && result < s )
			return "Unexpected end-of-file";
		
		return "Read error";
	}
	
	return 0;
}

error_t Data_Reader::skip( long count )
{
	char buf [512];
	while ( count )
	{
		long n = sizeof buf;
		if ( n > count )
			n = count;
		count -= n;
		error_t err = read( buf, n );
		if ( err )
			return err;
	}
	return 0;
}

long File_Reader::remain() const { return size() - tell(); }

error_t File_Reader::skip( long n )
{
	assert( n >= 0 );
	return n ? seek( tell() + n ) : 0;
}

// Subset_Reader

Subset_Reader::Subset_Reader( Data_Reader* dr, long size )
{
	in = dr;
	remain_ = dr->remain();
	if ( remain_ > size )
		remain_ = size;
}

long Subset_Reader::remain() const { return remain_; }

long Subset_Reader::read_avail( void* p, long s )
{
	if ( s > remain_ )
		s = remain_;
	remain_ -= s;
	return in->read_avail( p, s );
}

// Mem_File_Reader

Mem_File_Reader::Mem_File_Reader( const void* p, long s ) :
	begin( (const char*) p ),
	size_( s )
{
	pos = 0;
}
	
long Mem_File_Reader::size() const { return size_; }

long Mem_File_Reader::read_avail( void* p, long s )
{
	long r = remain();
	if ( s > r )
		s = r;
	memcpy( p, begin + pos, s );
	pos += s;
	return s;
}

long Mem_File_Reader::tell() const { return pos; }

error_t Mem_File_Reader::seek( long n )
{
	if ( n > size_ )
		return "Tried to go past end of file";
	pos = n;
	return 0;
}

// Std_File_Reader

Std_File_Reader::Std_File_Reader() : file_( 0 ) { }

Std_File_Reader::~Std_File_Reader() { close(); }

error_t Std_File_Reader::open( const char* path )
{
	file_ = fopen( path, "rb" );
	if ( !file_ )
		return "Couldn't open file";
	return 0;
}

long Std_File_Reader::size() const
{
	long pos = tell();
	fseek( (FILE*) file_, 0, SEEK_END );
	long result = tell();
	fseek( (FILE*) file_, pos, SEEK_SET );
	return result;
}

long Std_File_Reader::read_avail( void* p, long s )
{
	return (long) fread( p, 1, s, (FILE*) file_ );
}

long Std_File_Reader::tell() const { return ftell( (FILE*) file_ ); }

error_t Std_File_Reader::seek( long n )
{
	if ( fseek( (FILE*) file_, n, SEEK_SET ) != 0 )
		return "Error seeking in file";
	return 0;
}

void Std_File_Reader::close()
{
	if ( file_ )
	{
		fclose( (FILE*) file_ );
		file_ = 0;
	}
}

// Gzip_File_Reader

#ifdef HAVE_ZLIB_H

#include "zlib.h"

static const char* get_gzip_eof( const char* path, long* eof )
{
	FILE* file = fopen( path, "rb" );
	if ( !file )
		return "Couldn't open file";
	
	unsigned char buf [4];
	if ( fread( buf, 2, 1, file ) == 1 && buf [0] == 0x1F && buf [1] == 0x8B )
	{
		fseek( file, -4, SEEK_END );
		fread( buf, 4, 1, file );
		*eof = buf [3] * 0x1000000L + buf [2] * 0x10000L + buf [1] * 0x100L + buf [0];
	}
	else
	{
		fseek( file, 0, SEEK_END );
		*eof = ftell( file );
	}
	const char* err = (ferror( file ) || feof( file )) ? "Couldn't get file size" : 0;
	fclose( file );
	return err;
}

Gzip_File_Reader::Gzip_File_Reader() : file_( 0 ) { }

Gzip_File_Reader::~Gzip_File_Reader() { close(); }

error_t Gzip_File_Reader::open( const char* path )
{
	close();
	
	error_t err = get_gzip_eof( path, &size_ );
	if ( err )
		return err;
	
	file_ = gzopen( path, "rb" );
	if ( !file_ )
		return "Couldn't open file";
	
	return 0;
}

long Gzip_File_Reader::size() const { return size_; }

long Gzip_File_Reader::read_avail( void* p, long s ) { return (long) gzread( file_, p, s ); }

long Gzip_File_Reader::tell() const { return gztell( file_ ); }

error_t Gzip_File_Reader::seek( long n )
{
	if ( gzseek( file_, n, SEEK_SET ) < 0 )
		return "Error seeking in file";
	return 0;
}

void Gzip_File_Reader::close()
{
	if ( file_ )
	{
		gzclose( file_ );
		file_ = 0;
	}
}

#endif
